import { S3Client, PutObjectCommand } from "@aws-sdk/client-s3";
import { DynamoDBClient } from "@aws-sdk/client-dynamodb";
import { DynamoDBDocumentClient, PutCommand } from "@aws-sdk/lib-dynamodb";
import crypto from "crypto";


// Initialize AWS clients for Mumbai region
const s3Client = new S3Client({ region: "ap-south-1" });
const dynamoClient = new DynamoDBClient({ region: "ap-south-1" });
const docClient = DynamoDBDocumentClient.from(dynamoClient);

const S3_BUCKET = "temp-hasher-autho";
const DYNAMODB_TABLE = "hashes";
const ENCRYPTION_KEY = "g576f42xmoxofuvv3tdqribj2a0jdsgb";
const GEMINI_API_KEY = "AIzaSyC103mt4aE2LudFcT7qmbhuC8RgGW95Xho";

/**
 * Encrypt text using AES-256-CBC
 */
function encryptText(text, key) {
  const algorithm = 'aes-256-cbc';
  const keyBuffer = crypto.createHash('sha256').update(key).digest();
  const iv = crypto.randomBytes(16);
  const cipher = crypto.createCipheriv(algorithm, keyBuffer, iv);
  
  let encrypted = cipher.update(text, 'utf8', 'hex');
  encrypted += cipher.final('hex');
  
  return {
    encrypted,
    iv: iv.toString('hex')
  };
}

/**
 * Decrypt text using AES-256-CBC
 */
function decryptText(encryptedText, key, iv) {
  const algorithm = 'aes-256-cbc';
  const keyBuffer = crypto.createHash('sha256').update(key).digest();
  const ivBuffer = Buffer.from(iv, 'hex');
  const decipher = crypto.createDecipheriv(algorithm, keyBuffer, ivBuffer);
  
  let decrypted = decipher.update(encryptedText, 'hex', 'utf8');
  decrypted += decipher.final('utf8');
  
  return decrypted;
}

/**
 * Calculate content match and detect differences
 */
function analyzeContentDifferences(originalText, uploadedText) {
  if (!originalText || !uploadedText) return { matchPercentage: 0, differences: [] };
  
  const normalize = (text) => text.toLowerCase().replace(/\s+/g, ' ').trim();
  const norm1 = normalize(originalText);
  const norm2 = normalize(uploadedText);
  
  if (norm1 === norm2) return { matchPercentage: 100, differences: [] };
  
  const words1 = norm1.split(' ');
  const words2 = norm2.split(' ');
  
  const totalWords = Math.max(words1.length, words2.length);
  if (totalWords === 0) return { matchPercentage: 0, differences: [] };
  
  let matchedWords = 0;
  const words2Set = new Set(words2);
  const differences = [];
  
  // Find missing words from original
  for (const word of words1) {
    if (words2Set.has(word)) {
      matchedWords++;
    } else {
      differences.push({ type: 'removed', word });
    }
  }
  
  // Find added words in uploaded
  const words1Set = new Set(words1);
  for (const word of words2) {
    if (!words1Set.has(word)) {
      differences.push({ type: 'added', word });
    }
  }
  
  const matchPercentage = Math.round((matchedWords / totalWords) * 100);
  
  return { matchPercentage, differences: differences.slice(0, 20) }; // Limit to 20 differences
}

/**
 * Get AI analysis from Gemini with fallback
 */
async function getGeminiAnalysis(matchPercentage, differences, originalLength, uploadedLength) {
  // Fallback analysis if API fails
  const fallbackAnalysis = () => {
    if (matchPercentage === 100) {
      return 'Document shows 100% content match with original. This indicates high integrity and authenticity. The document can be trusted for secure operations.';
    } else if (matchPercentage >= 90) {
      return `Document shows ${matchPercentage}% match with minor differences (${differences.length} changes). Generally trustworthy but review changes carefully.`;
    } else if (matchPercentage >= 70) {
      return `Document shows ${matchPercentage}% match with moderate changes (${differences.length} differences). Exercise caution and verify critical content before use.`;
    } else {
      return `Document shows only ${matchPercentage}% match with significant changes (${differences.length} differences). High risk of tampering - do not trust for sensitive operations.`;
    }
  };
  
  try {
    const { GoogleGenerativeAI } = await import('@google/generative-ai');
    const genAI = new GoogleGenerativeAI(GEMINI_API_KEY);
    const model = genAI.getGenerativeModel({ model: 'gemini-2.5-flash-lite' });
    
    let prompt;
    if (matchPercentage === 100) {
      prompt = `Analyze this PDF integrity check result: 100% content match found between original and uploaded document (${originalLength} characters). Provide a clean trust assessment in 2-3 sentences without any markdown formatting, asterisks, hyphens or special characters. Focus on document trustworthiness for secure operations.`;
    } else {
      prompt = `Analyze this PDF integrity check: ${matchPercentage}% content match found with ${differences.length} differences between original and uploaded document. Sample changes: ${differences.slice(0, 3).map(d => `${d.type} "${d.word}"`).join(', ')}. Provide a clean trust assessment in 2-3 sentences without markdown formatting. Should the user trust this document?`;
    }
    
    const result = await model.generateContent(prompt);
    const response = await result.response;
    const text = response.text();
    
    return text || fallbackAnalysis();
  } catch (error) {
    console.warn('Gemini API failed:', error.message);
    return fallbackAnalysis();
  }
}

/**
 * Embed hash into PDF using steganography (metadata injection)
 */
function embedHashInPDF(pdfBuffer, hash) {
  const pdfString = pdfBuffer.toString('binary');
  
  // Find the trailer section
  const trailerIndex = pdfString.lastIndexOf('trailer');
  if (trailerIndex === -1) {
    // Fallback: append to end before %%EOF
    const eofIndex = pdfString.lastIndexOf('%%EOF');
    if (eofIndex === -1) {
      return pdfBuffer; // Can't modify, return original
    }
    
    const beforeEof = pdfString.substring(0, eofIndex);
    const afterEof = pdfString.substring(eofIndex);
    const hiddenData = `\n% Hidden: ${Buffer.from(hash).toString('base64')}\n`;
    
    return Buffer.from(beforeEof + hiddenData + afterEof, 'binary');
  }
  
  // Insert hidden hash in trailer section
  const beforeTrailer = pdfString.substring(0, trailerIndex);
  const afterTrailer = pdfString.substring(trailerIndex);
  const hiddenData = `% Embedded: ${Buffer.from(hash).toString('base64')}\n`;
  
  return Buffer.from(beforeTrailer + hiddenData + afterTrailer, 'binary');
}

/**
 * Decode embedded hash from steganographed PDF
 */
function decodeHashFromPDF(pdfBuffer) {
  const pdfString = pdfBuffer.toString('binary');
  
  // Look for hidden data patterns
  const hiddenMatch = pdfString.match(/% Hidden: ([A-Za-z0-9+/=]+)/);
  if (hiddenMatch) {
    return Buffer.from(hiddenMatch[1], 'base64').toString();
  }
  
  const embeddedMatch = pdfString.match(/% Embedded: ([A-Za-z0-9+/=]+)/);
  if (embeddedMatch) {
    return Buffer.from(embeddedMatch[1], 'base64').toString();
  }
  
  return null;
}

/**
 * Lambda handler to accept PDF uploads from users, extract data, and store hash
 * 
 * Expects:
 * - Multipart form upload OR
 * - Base64 encoded PDF in request body: { file: "base64-string", fileName: "doc.pdf" }
 */
export const handler = async (event) => {
  const path = event.rawPath || event.path || '/';
  const method = event.requestContext?.http?.method || event.httpMethod || 'POST';
  
  // Route to decode endpoint
  if (path === '/decode' && method === 'POST') {
    return await handleDecode(event);
  }
  
  // Default upload route
  return await handleUpload(event);
};

/**
 * Handle PDF decode route
 */
async function handleDecode(event) {
  try {
    let pdfBuffer;
    
    if (event.body) {
      const isBase64Encoded = event.isBase64Encoded || false;
      
      if (typeof event.body === 'string') {
        if (isBase64Encoded) {
          pdfBuffer = Buffer.from(event.body, 'base64');
        } else {
          const body = JSON.parse(event.body);
          if (body.file) {
            pdfBuffer = Buffer.from(body.file, 'base64');
          } else {
            throw new Error('No file data found');
          }
        }
      }
    }
    
    if (!pdfBuffer || !isPDF(pdfBuffer)) {
      throw new Error('Invalid PDF file');
    }
    
    const embeddedHash = decodeHashFromPDF(pdfBuffer);
    
    // Extract text from uploaded PDF for comparison
    let uploadedText = '';
    try {
      const { default: pdf } = await import('pdf-parse');
      const data = await pdf(pdfBuffer);
      uploadedText = data.text || '';
    } catch (pdfError) {
      console.warn('PDF text extraction failed:', pdfError.message);
      uploadedText = '';
    }
    
    let dbRecord = null;
    let existsInDB = false;
    let contentMatchPercentage = 0;
    
    if (embeddedHash) {
      // Check if hash exists in DynamoDB
      try {
        const { GetCommand } = await import('@aws-sdk/lib-dynamodb');
        const result = await docClient.send(new GetCommand({
          TableName: DYNAMODB_TABLE,
          Key: { cryp: embeddedHash }
        }));
        
        if (result.Item) {
          dbRecord = result.Item;
          existsInDB = true;
          
          // Decrypt the encrypted text if found
          if (dbRecord.encryptedText && dbRecord.encryptionIV) {
            try {
              const decryptedText = decryptText(dbRecord.encryptedText, ENCRYPTION_KEY, dbRecord.encryptionIV);
              dbRecord.decryptedText = decryptedText;
            } catch (decryptError) {
              console.warn('Decryption failed:', decryptError.message);
              dbRecord.decryptedText = 'Decryption failed';
            }
          }
          
          // Analyze content differences
          let analysisResult = { matchPercentage: 0, differences: [] };
          let aiAnalysis = '';
          
          if (dbRecord.decryptedText && uploadedText) {
            analysisResult = analyzeContentDifferences(dbRecord.decryptedText, uploadedText);
            contentMatchPercentage = analysisResult.matchPercentage;
            
            // Always get AI analysis
            aiAnalysis = await getGeminiAnalysis(
              contentMatchPercentage,
              analysisResult.differences,
              dbRecord.decryptedText.length,
              uploadedText.length
            );
          }
          
          // Clean up AI response
          const cleanAiAnalysis = aiAnalysis
            .replace(/\*\*/g, '')
            .replace(/\*/g, '')
            .replace(/[-•]/g, '')
            .replace(/\n\s*\n/g, ' ')
            .replace(/\s+/g, ' ')
            .trim();
          
          dbRecord.contentAnalysis = {
            matchPercentage: contentMatchPercentage,
            totalDifferences: analysisResult.differences.length,
            contentChanges: {
              wordsRemoved: analysisResult.differences.filter(d => d.type === 'removed').map(d => d.word),
              wordsAdded: analysisResult.differences.filter(d => d.type === 'added').map(d => d.word)
            },
            trustAssessment: cleanAiAnalysis,
            recommendation: contentMatchPercentage === 100 ? 'TRUSTED' : 
                          contentMatchPercentage >= 90 ? 'MOSTLY_TRUSTED' :
                          contentMatchPercentage >= 70 ? 'CAUTION_REQUIRED' : 'NOT_TRUSTED'
          };
        }
      } catch (dbError) {
        console.warn('Database lookup failed:', dbError.message);
      }
    }
    
    return {
      statusCode: 200,
      body: JSON.stringify({
        success: true,
        embeddedHash: embeddedHash,
        found: !!embeddedHash,
        existsInDatabase: existsInDB,
        contentMatchPercentage: contentMatchPercentage,
        uploadedTextLength: uploadedText.length,
        record: dbRecord
      })
    };
    
  } catch (error) {
    return {
      statusCode: 400,
      body: JSON.stringify({
        success: false,
        error: error.message
      })
    };
  }
}

/**
 * Handle PDF upload route
 */
async function handleUpload(event) {
  try {
    let pdfBuffer;
    let fileName = 'uploaded.pdf';
    let userEmail = null;
    
    // Get required email from query parameters
    if (!event.queryStringParameters?.email) {
      throw new Error('Email is required. Please provide email parameter.');
    }
    userEmail = event.queryStringParameters.email;
    
    // Handle API Gateway request
    if (event.body) {
      const isBase64Encoded = event.isBase64Encoded || false;
      let body;
      
      // Parse the body
      if (typeof event.body === 'string') {
        // If it's base64 encoded binary data (direct file upload)
        if (isBase64Encoded) {
          pdfBuffer = Buffer.from(event.body, 'base64');
          
          // Try to get filename from headers
          const contentType = event.headers?.['content-type'] || event.headers?.['Content-Type'] || '';
          if (contentType.includes('multipart/form-data')) {
            const match = event.body.match(/filename="([^"]+)"/);
            if (match) fileName = match[1];
          }
          
          // Or from query parameters
          if (event.queryStringParameters?.fileName) {
            fileName = event.queryStringParameters.fileName;
          }
        } else {
          // JSON body with base64 file
          try {
            body = JSON.parse(event.body);
            if (body.file) {
              pdfBuffer = Buffer.from(body.file, 'base64');
              fileName = body.fileName || fileName;
            } else {
              throw new Error('No file data found in request body');
            }
          } catch (parseError) {
            throw new Error('Invalid JSON body or missing file data');
          }
        }
      } else {
        // Body is already an object
        body = event.body;
        if (body.file) {
          pdfBuffer = Buffer.from(body.file, 'base64');
          fileName = body.fileName || fileName;
        }
      }
    } else {
      throw new Error('No file uploaded. Please provide a PDF file.');
    }

    if (!pdfBuffer || pdfBuffer.length === 0) {
      throw new Error('Empty or invalid PDF file');
    }

    // Validate PDF
    if (!isPDF(pdfBuffer)) {
      throw new Error('Uploaded file is not a valid PDF');
    }

    // Generate hash of PDF content
    const pdfHash = crypto.createHash('sha256').update(pdfBuffer).digest('hex');
    const timestamp = new Date().toISOString();
    
    // Generate S3 key with hash and timestamp
    const s3Key = `uploads/${timestamp.split('T')[0]}/${pdfHash}-${fileName}`;
    
    // Upload to S3
    await s3Client.send(new PutObjectCommand({
      Bucket: S3_BUCKET,
      Key: s3Key,
      Body: pdfBuffer,
      ContentType: 'application/pdf',
      Metadata: {
        uploadedAt: timestamp,
        originalName: fileName,
        hash: pdfHash
      }
    }));
    
    console.log(`Uploaded PDF to S3: ${s3Key}`);
    
    // Extract text from PDF
    let extractedText = '';
    let pages = 1;
    try {
      const { default: pdf } = await import('pdf-parse');
      const data = await pdf(pdfBuffer);
      
      extractedText = data.text || 'No text found';
      pages = data.numpages || 1;
      
      console.log(`Extracted ${extractedText.length} characters from ${pages} pages`);
      
    } catch (pdfError) {
      console.warn('PDF text extraction failed:', pdfError.message);
      extractedText = 'Text extraction failed';
    }
    
    // Encrypt the extracted text
    const encryptedData = encryptText(extractedText, ENCRYPTION_KEY);
    
    // Embed hash into PDF using steganography
    const steganographedPDF = embedHashInPDF(pdfBuffer, pdfHash);
    console.log(`Embedded hash into PDF using steganography`);
    
    // Upload steganographed PDF to S3
    const stegoS3Key = `files/returned/${pdfHash}-stego-${fileName}`;
    await s3Client.send(new PutObjectCommand({
      Bucket: S3_BUCKET,
      Key: stegoS3Key,
      Body: steganographedPDF,
      ContentType: 'application/pdf',
      Metadata: {
        originalHash: pdfHash,
        steganographed: 'true',
        createdAt: timestamp
      }
    }));
    
    const stegoFileUrl = `https://${S3_BUCKET}.s3.ap-south-1.amazonaws.com/${stegoS3Key}`;
    console.log(`Uploaded steganographed PDF to S3: ${stegoS3Key}`);
    
    // Prepare DynamoDB item
    const dynamoItem = {
      cryp: pdfHash,
      timestamp,
      fileName,
      s3Key,
      s3Bucket: S3_BUCKET,
      fileSize: pdfBuffer.length,
      pages,
      extractedAt: timestamp,
      source: 'user_upload',
      textLength: extractedText.length,
      textPreview: extractedText.substring(0, 500),
      encryptedText: encryptedData.encrypted,
      encryptionIV: encryptedData.iv
    };
    
    // Add required email
    dynamoItem.userEmail = userEmail;
    
    // Store hash and metadata in DynamoDB
    await docClient.send(new PutCommand({
      TableName: DYNAMODB_TABLE,
      Item: dynamoItem
    }));
    
    console.log(`Stored hash in DynamoDB: ${pdfHash}`);
    
    // Return success response
    return {
      statusCode: 200,
      body: JSON.stringify({
        success: true,
        message: 'PDF uploaded and processed successfully',
        hash: pdfHash,
        s3Location: {
          bucket: S3_BUCKET,
          key: s3Key
        },
        data: {
          fileName,
          fileSize: pdfBuffer.length,
          pages,
          textLength: extractedText.length,
          extractedText,
          encryptedText: encryptedData.encrypted,
          steganographedPDFUrl: stegoFileUrl
        },
        dynamodb: {
          table: DYNAMODB_TABLE,
          partitionKey: pdfHash
        }
      })
    };
    
  } catch (error) {
    console.error('Error processing PDF upload:', error);
    return {
      statusCode: error.message.includes('not a valid PDF') ? 400 : 500,
      body: JSON.stringify({
        success: false,
        error: error.message,
        stack: process.env.NODE_ENV === 'development' ? error.stack : undefined
      })
    };
  }
};

/**
 * Validate if buffer is a PDF file
 */
function isPDF(buffer) {
  // PDF files start with "%PDF-"
  const header = buffer.slice(0, 5).toString();
  return header === '%PDF-';
}

/**
 * Helper function to convert stream to buffer
 */
async function streamToBuffer(stream) {
  const chunks = [];
  for await (const chunk of stream) {
    chunks.push(chunk);
  }
  return Buffer.concat(chunks);
}